## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.path = "figures/sobol-exploring-",
  fig.width = 6,
  fig.height = 4.5,
  dpi = 150,
  message = FALSE,
  warning = FALSE,
  eval=FALSE
)

LOCAL <- identical(Sys.getenv("LOCAL"), "TRUE")

library(sensitivity)
library(Sobol4R)
set.seed(4669)

## ----det-design, cache=TRUE, eval=LOCAL---------------------------------------
n <- 50000
p <- 8
X1_1 <- data.frame(matrix(runif(p * n), nrow = n))
X2_1 <- data.frame(matrix(runif(p * n), nrow = n))

## ----det-g-run, cache=TRUE, eval=LOCAL----------------------------------------
set.seed(4669)
gensol1 <- sobol4r_design(
  X1    = X1_1,
  X2    = X2_1,
  order = 2,
  nboot = 100
)

Y1 <- sobol_g_function(gensol1$X)
x1 <- sensitivity::tell(gensol1, Y1)

## ----det-g-plot, cache=TRUE, eval=LOCAL---------------------------------------
print(x1)
Sobol4R::autoplot(x1, ncol = 1)

## ----echo=FALSE, cache=TRUE, eval=LOCAL---------------------------------------
rm(gensol1, X1_1, X2_1)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
ex1_results <- sobol_example_g_deterministic()
print(ex1_results)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
Sobol4R::autoplot(ex1_results, ncol = 1)

## ----echo=FALSE, cache=TRUE, eval=LOCAL---------------------------------------
rm(ex1_results)

## ----r1-design, cache=TRUE, eval=LOCAL----------------------------------------
n <- 50000
X1_r1 <- data.frame(
  C1 = runif(n),
  C2 = runif(n)
)
X2_r1 <- data.frame(
  C1 = runif(n),
  C2 = runif(n)
)

## ----r1-sobol-design, cache=TRUE, eval=LOCAL----------------------------------
set.seed(4669)
gensol2 <- sobol4r_design(
  X1    = X1_r1,
  X2    = X2_r1,
  order = 2,
  nboot = 100
)

## ----r1-Y, cache=TRUE, eval=LOCAL---------------------------------------------
Y2 <- sobol_g2_function(gensol2$X)
Y3 <- sobol_g2_additive_noise(gensol2$X)
Y4 <- sobol_g2_qoi_mean(gensol2$X, nrep = 1000)

## ----r1-results, cache=TRUE, eval=LOCAL---------------------------------------
x2 <- sensitivity::tell(gensol2, Y2)
x3 <- sensitivity::tell(gensol2, Y3)
x4 <- sensitivity::tell(gensol2, Y4)

## ----r1-print, cache=TRUE, eval=LOCAL-----------------------------------------
print(x2)
print(x3)
print(x4)

## ----r1-plot, fig.keep='all', cache=TRUE, eval=LOCAL--------------------------
Sobol4R::autoplot(x2)
Sobol4R::autoplot(x3)
Sobol4R::autoplot(x4)

## ----echo=FALSE, cache=TRUE, eval=LOCAL---------------------------------------
rm(gensol2)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
ex2_results <- sobol_example_random_output()
ex2_results

## ----fig.keep='all', cache=TRUE, eval=LOCAL-----------------------------------
Sobol4R::autoplot(ex2_results$x_det)
Sobol4R::autoplot(ex2_results$x_noise)
Sobol4R::autoplot(ex2_results$x_qoi)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
rm(ex2_results)

## ----r2-design, cache=TRUE, eval=LOCAL----------------------------------------
n <- 50000
X1_r2 <- data.frame(
  C1 = X1_r1$C1,
  C2 = X1_r1$C2,
  C3 = runif(n, min = 1, max = 100)
)
X2_r2 <- data.frame(
  C1 = X2_r1$C1,
  C2 = X2_r1$C2,
  C3 = runif(n, min = 1, max = 100)
)

## ----r2-head, cache=TRUE, eval=LOCAL------------------------------------------
head(X1_r1)
head(X1_r2)

## ----r2-sobol-design, cache=TRUE, eval=LOCAL----------------------------------
set.seed(4669)
gensol3 <- sobol4r_design(
  X1    = X1_r2,
  X2    = X2_r2,
  order = 2,
  nboot = 100
)

## ----r2-Y, cache=TRUE, eval=LOCAL---------------------------------------------
Y5 <- sobol_g2_with_covariate_noise(gensol3$X)
Y6 <- sobol_g2_qoi_covariate_mean(gensol3$X, nrep = 1000)

## ----r2-results, cache=TRUE, eval=LOCAL---------------------------------------
x5 <- sensitivity::tell(gensol3, Y5)
x6 <- sensitivity::tell(gensol3, Y6)

## ----r2-print, cache=TRUE, eval=LOCAL-----------------------------------------
print(x5)
print(x6)

## ----r2-plot, fig.keep='all', cache=TRUE, eval=LOCAL--------------------------
Sobol4R::autoplot(x5)
Sobol4R::autoplot(x6)

## ----echo=FALSE, cache=TRUE, eval=LOCAL---------------------------------------
rm(gensol3, X1_r2, X2_r2)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
ex3_results <- sobol_example_covariate_large()
ex3_results

## ----fig.keep='all', cache=TRUE, eval=LOCAL-----------------------------------
Sobol4R::autoplot(ex3_results$x_single)
Sobol4R::autoplot(ex3_results$x_qoi)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
rm(ex3_results)

## ----r3-design, cache=TRUE, eval=LOCAL----------------------------------------
n <- 50000
X1_r3 <- data.frame(
  C1 = X1_r1$C1,
  C2 = X1_r1$C2,
  C3 = runif(n, min = 1, max = 1.5)
)
X2_r3 <- data.frame(
  C1 = X2_r1$C1,
  C2 = X2_r1$C2,
  C3 = runif(n, min = 1, max = 1.5)
)

## ----r3-sobol-design, cache=TRUE, eval=LOCAL----------------------------------
set.seed(4669)
gensol4 <- sobol4r_design(
  X1    = X1_r3,
  X2    = X2_r3,
  order = 2,
  nboot = 100
)

## ----r3-Y, cache=TRUE, eval=LOCAL---------------------------------------------
Y7 <- sobol_g2_with_covariate_noise(gensol4$X)
Y8 <- sobol_g2_qoi_covariate_mean(gensol4$X, nrep = 1000)

## ----r3-results, cache=TRUE, eval=LOCAL---------------------------------------
x7 <- sensitivity::tell(gensol4, Y7)
x8 <- sensitivity::tell(gensol4, Y8)

## ----r3-print, cache=TRUE, eval=LOCAL-----------------------------------------
print(x7)
print(x8)

## ----r3-plot, fig.keep='all', cache=TRUE, eval=LOCAL--------------------------
Sobol4R::autoplot(x7)
Sobol4R::autoplot(x8)

## ----echo=FALSE, cache=TRUE, eval=LOCAL---------------------------------------
rm(gensol4, X1_r3, X2_r3)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
ex4_results <- sobol_example_covariate_small()
ex4_results

## ----fig.keep='all', cache=TRUE, eval=LOCAL-----------------------------------
Sobol4R::autoplot(ex4_results$x_single)
Sobol4R::autoplot(ex4_results$x_qoi)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
rm(ex4_results)

## ----process-design, cache=TRUE, eval=LOCAL-----------------------------------
n <- 100

draw_params <- function(n) {
  data.frame(t(replicate(
    n,
    c(
      1 / runif(1, min = 20,  max = 100),
      1 / runif(1, min = 24,  max = 2000),
      1 / runif(1, min = 24,  max = 120),
      runif(1,  min = 0.05, max = 0.3),
      runif(1,  min = 0.3,  max = 0.7)
    )
  )))
}

X1_process <- draw_params(n)
X2_process <- draw_params(n)

## ----process-sobol-design, cache=TRUE, eval=LOCAL-----------------------------
set.seed(4669)
gensolp1 <- sobol4r_design(
  X1    = X1_process,
  X2    = X2_process,
  order = 2,
  nboot = 10
)

## ----process-Y, cache=TRUE, eval=LOCAL----------------------------------------
MM <- 50

Yp1 <- process_fun_row_wise(gensolp1$X, M = MM)
Yp2 <- process_fun_mean_to_M(gensolp1$X, M = MM, nrep = 10)

## ----process-results, cache=TRUE, eval=LOCAL----------------------------------
xp1 <- sensitivity::tell(gensolp1, Yp1)
xp2 <- sensitivity::tell(gensolp1, Yp2)

## ----process-print, cache=TRUE, eval=LOCAL------------------------------------
print(xp1)
print(xp2)

## ----process-plot, fig.keep='all', cache=TRUE, eval=LOCAL---------------------
Sobol4R::autoplot(xp1)
Sobol4R::autoplot(xp2)

## ----echo=FALSE, cache=TRUE, eval=LOCAL---------------------------------------
rm(
  X1_r1, X2_r1,
  X1_process, X2_process,
  gensolp1
)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
ex5_results <- sobol_example_process(order = 2)
ex5_results

## ----fig.keep='all', cache=TRUE, eval=LOCAL-----------------------------------
Sobol4R::autoplot(ex5_results$xp_single)
Sobol4R::autoplot(ex5_results$xp_qoi)

## ----cache=TRUE, eval=LOCAL---------------------------------------------------
rm(ex5_results)

